//////////////////////////////////////////////////////////////
//             misc matrix and camera funcs                 //
//////////////////////////////////////////////////////////////

//
//  Note: for a given transform (M,T) we always store the direct
//  and well as the reciprocal matrix within the same 2*(9+3) array
//  of floats. Entries [0-8] is the matrix M, entries [9-17] is iM
//  (the inverse), entries [18-20] is the translation T, and entries
//  [21-23] is the 'reciprocal' translation -iM.T (beware of sign).
//  Composing two such matrix requires special care for this format
//  being respected all along the chain of composition (see function
//  Mult_Matrix2_BA() for instance)...
//

typedef float  VECTOR[3];
typedef float  MATRIX[9+9];       // 9 for direct, 9 for inv
typedef float  MATRIX2[9+9+3+3];  // 9 for direct, 9 for inv, 3+3 for translation
typedef float  MATRIX3x3[9];
#define MATRIX_INV(M)         ((float*)&(M)[9])
#define MATRIX_TRANSL(M)      ((float*)&(M)[18])
#define MATRIX_INV_TRANSL(M)  ((float*)&(M)[21])

//////////////////////////////////////////////////////////////

         // useful macros...

#define Assign_Vector(A,B) { (A)[0]=(B)[0];(A)[1]=(B)[1];(A)[2]=(B)[2]; }
#define Set_Vector(V,x,y,z) { (V)[0] = (x); (V)[1] = (y); (V)[2] = (z); }
#define Add_Vector(A,B,C) { (A)[0] = (B)[0]+(C)[0]; (A)[1] = (B)[1]+(C)[1]; (A)[2] = (B)[2]+(C)[2]; }
#define Add_Vector_Eq(A,B) { (A)[0] += (B)[0]; (A)[1] += (B)[1]; (A)[2] += (B)[2]; }
#define Scale_Vector_Eq(A,B) { (A)[0]*=(B);(A)[1]*=(B);(A)[2]*=(B); }
#define Scale_Vector(A,B,C) { (A)[0]=(C)*(B)[0]; (A)[1]=(C)*(B)[1];(A)[2]=(C)*(B)[2]; }
#define Add_Scaled_Vector_Eq(A,B,C) { (A)[0]+=(C)*(B)[0]; (A)[1]+=(C)*(B)[1];(A)[2]+=(C)*(B)[2]; }
#define Add_Scaled_Vector(A,B,C,D) { (A)[0]=(B)[0]+(D)*(C)[0];(A)[1]=(B)[1]+(D)*(C)[1];(A)[2]=(B)[2]+(D)*(C)[2];}

#define Cross_Product(A,B,C) {        \
   (A)[0] = (B)[1]*(C)[2] - (B)[2]*(C)[1];    \
   (A)[1] = (B)[2]*(C)[0] - (B)[0]*(C)[2];    \
   (A)[2] = (B)[0]*(C)[1] - (B)[1]*(C)[0];    \
}

#define Dot_Product(B,C) ( (B)[0]*(C)[0] + (B)[1]*(C)[1] +  (B)[2]*(C)[2] )
#define Norm_Squared(A) ( Dot_Product((A),(A)) )

//////////////////////////////////////////////////////////////


static void A_V( VECTOR Out, MATRIX2 M, VECTOR In )
{
      // Out = M.In + T
      // beware of self-modifying problem if Out==In... 
   VECTOR Tmp;

   Tmp[0] = ( M[0]*In[0] + M[1]*In[1] + M[2]*In[2] ) + M[18];
   Tmp[1] = ( M[3]*In[0] + M[4]*In[1] + M[5]*In[2] ) + M[19];
   Tmp[2] = ( M[6]*In[0] + M[7]*In[1] + M[8]*In[2] ) + M[20];
   Assign_Vector( Out, Tmp );
}
#define A_V_Eq(V,M)  A_V(V,M,V)

static void nA_V( VECTOR Out, MATRIX M, VECTOR In )
{
      // Out = t(M^-1).In
   VECTOR Tmp;
   Tmp[0] = ( M[ 9]*In[0] + M[12]*In[1] + M[15]*In[2] );
   Tmp[1] = ( M[10]*In[0] + M[13]*In[1] + M[16]*In[2] );
   Tmp[2] = ( M[11]*In[0] + M[14]*In[1] + M[17]*In[2] );
   Assign_Vector( Out, Tmp );
}
#define nA_V_Eq(V,M)  nA_V(V,M,V)

static void A_Inv_V( VECTOR Out, MATRIX M, VECTOR In )
{
      // Out = iM.In - iM.T
   VECTOR Tmp;
   Tmp[0] = ( M[ 9]*In[0] + M[10]*In[1] + M[11]*In[2] ) + M[21];
   Tmp[1] = ( M[12]*In[0] + M[13]*In[1] + M[14]*In[2] ) + M[22];
   Tmp[2] = ( M[15]*In[0] + M[16]*In[1] + M[17]*In[2] ) + M[23];
   Assign_Vector( Out, Tmp );
}
#define A_Inv_V_Eq(V,M)  A_Inv_V(V,M,V)

static void nA_Inv_V( VECTOR Out, MATRIX M, VECTOR In )
{
      // Out = tM.In
   VECTOR Tmp;
   Tmp[0] = ( M[0]*In[0] + M[3]*In[1] + M[6]*In[2] );
   Tmp[1] = ( M[1]*In[0] + M[4]*In[1] + M[7]*In[2] );
   Tmp[2] = ( M[2]*In[0] + M[5]*In[1] + M[8]*In[2] );
   Assign_Vector( Out, Tmp );
}
#define nA_Inv_V_Eq(V,M)  nA_Inv_V(V,M,V)


static void Mult_Matrix_3x3( MATRIX3x3 A, MATRIX3x3 B, MATRIX3x3 C )
{
      /* A = B.C */
   A[0] = B[0]*C[0] + B[1]*C[3] + B[2]*C[6];
   A[1] = B[0]*C[1] + B[1]*C[4] + B[2]*C[7];
   A[2] = B[0]*C[2] + B[1]*C[5] + B[2]*C[8];
   A[3] = B[3]*C[0] + B[4]*C[3] + B[5]*C[6];
   A[4] = B[3]*C[1] + B[4]*C[4] + B[5]*C[7];
   A[5] = B[3]*C[2] + B[4]*C[5] + B[5]*C[8];
   A[6] = B[6]*C[0] + B[7]*C[3] + B[8]*C[6];
   A[7] = B[6]*C[1] + B[7]*C[4] + B[8]*C[7];
   A[8] = B[6]*C[2] + B[7]*C[5] + B[8]*C[8];
}

static void Mult_Matrix( MATRIX A, MATRIX B )
{
   MATRIX C;

      /* A.B=>B */

   Mult_Matrix_3x3( C, A, B );
   Mult_Matrix_3x3( MATRIX_INV(C), MATRIX_INV(B), MATRIX_INV(A) );
   memcpy( B, C, sizeof( MATRIX ) );
}

static void Mult_Matrix2_AB( MATRIX2 A, MATRIX2 B )
{
   MATRIX C;
   VECTOR V;
      /* B = A.B */
   Mult_Matrix_3x3( C, A, B );
   Mult_Matrix_3x3( MATRIX_INV(C), MATRIX_INV(B), MATRIX_INV(A) );

   A_V_Eq( MATRIX_TRANSL(B), A );            // Tb = A.Tb + Ta
   A_Inv_V( V, B, MATRIX_INV_TRANSL(A) );    // V = B^-1.( -(A^-1)Ta ) -(B^-1).Tb
   Assign_Vector( MATRIX_INV_TRANSL(B), V ); // => -(AB)^-1.Ta - B^-1.Tb
   memcpy( B, C, sizeof( MATRIX )  );
}

static void Mult_Matrix2_BA( MATRIX2 A, MATRIX2 B )
{
   MATRIX C;
   VECTOR V;

      /* A = A.B */
   Mult_Matrix_3x3( C, A, B );
   Mult_Matrix_3x3( MATRIX_INV(C), MATRIX_INV(B), MATRIX_INV(A) );

   A_Inv_V_Eq( MATRIX_INV_TRANSL(A), B );
   A_V( V, A, MATRIX_TRANSL(B) );         // Tb = A.Tb + Ta
   Assign_Vector( MATRIX_TRANSL(A), V );  // => -(AB)^-1.Ta - B^-1.Tb
   memcpy( A, C, sizeof( MATRIX ) );
}

static void Mult_Matrix_ABC( MATRIX2 A, MATRIX2 B, MATRIX2 C )
{
      /* C = A.B */
   Mult_Matrix_3x3( C, A, B );
   Mult_Matrix_3x3( MATRIX_INV(C), MATRIX_INV(B), MATRIX_INV(A) );

   A_V( MATRIX_TRANSL(C), A, MATRIX_TRANSL(B) );
   A_Inv_V( MATRIX_INV_TRANSL(C), B, MATRIX_INV_TRANSL(A) );
}

static void Id_Matrix( MATRIX M )
{
   M[9]  = M[0] = M[13] = M[4] = M[17] = M[8] = 1.0;
   M[10] = M[1] = M[11] = M[2] = M[12] = M[3] = 0.0;
   M[14] = M[5] = M[15] = M[6] = M[16] = M[7] = 0.0;
}

static void Id_Matrix2( MATRIX2 M )
{
   Id_Matrix( M );
   M[18] = M[19] = M[20] = 0.0;
   M[21] = M[22] = M[23] = 0.0;
}

static void Scale_Matrix( MATRIX2 M, float Sx, float Sy, float Sz )
{
   Id_Matrix2( M );

   if ( Sx==0.0 ) Sx = 1.0f;     // sure-fire hack...
   if ( Sy==0.0 ) Sy = 1.0f;
   if ( Sz==0.0 ) Sz = 1.0f;

   M[0] = Sx; M[4] = Sy; M[8] = Sz;
   M[9] = 1.0f/Sx; M[13] = 1.0f/Sy; M[17] = 1.0f/Sz;
}

//////////////////////////////////////////////////////////////

      // computes the matrix for a rotation of <Alpha> radians
      // around the vector <Axis>.

static void Axis_Rotation( VECTOR Axis, float Alpha, MATRIX2 M )
{
   float l, Cx, Sx;
   VECTOR D;

   l  = Axis[0]*Axis[0];
   l += Axis[1]*Axis[1];
   l += Axis[2]*Axis[2];
   if ( l==0.0 ) { Id_Matrix( M ); return; }
   l = (float)( 1.0/sqrt( l ) );
   D[0] = Axis[0]*l; 
   D[1] = Axis[1]*l; 
   D[2] = Axis[2]*l;

   Cx = (float)cos( Alpha ); Sx = (float)sin( Alpha );

   M[0] = M[9]  = D[0]*D[0]*(1.0f-Cx) + Cx;
   M[1] = M[12] = D[0]*D[1]*(1.0f-Cx) + D[2]*Sx;
   M[2] = M[15] = D[0]*D[2]*(1.0f-Cx) - D[1]*Sx;

   M[3] = M[10] = D[0]*D[1]*(1.0f-Cx) - D[2]*Sx;
   M[4] = M[13] = D[1]*D[1]*(1.0f-Cx) + Cx;
   M[5] = M[16] = D[1]*D[2]*(1.0f-Cx) + D[0]*Sx;

   M[6] = M[11] = D[0]*D[2]*(1.0f-Cx) + D[1]*Sx;
   M[7] = M[14] = D[1]*D[2]*(1.0f-Cx) - D[0]*Sx;
   M[8] = M[17] = D[2]*D[2]*(1.0f-Cx) + Cx;

   M[18] = M[19] = M[20] = 0.0;
   M[21] = M[22] = M[23] = 0.0;
}

      // Compute camera matrix from yaw, pitch, roll and position

static void Camera_RYP( float Yaw, float Pitch, float Roll, VECTOR Pos, MATRIX2 M )
{
   double cy, sy, cr, sr, cp, sp, A, B;

   cy = cos( Yaw );   sy = sin( Yaw );
   cp = cos( Pitch ); sp = sin( Pitch );
   cr = cos( Roll );  sr = sin( Roll );

   A = sy*sr;
   B = cy*sr;

   M[9]  = M[0] = (float)( cy*cr );
   M[12] = M[1] = (float)(-sr );
   M[15] = M[2] = (float)(-sy*cr );

   M[10] = M[3] = (float)( B*cp + sy*sp );
   M[13] = M[4] = (float)( cr*cp );
   M[16] = M[5] = (float)(-A*cp + cy*sp );

   M[11] = M[6] = (float)(-B*sp + sy*cp );
   M[14] = M[7] = (float)(-cr*sp );
   M[17] = M[8] = (float)( A*sp + cy*cp );

   MATRIX_INV_TRANSL(M)[0] =  Pos[0];
   MATRIX_TRANSL(M)[0]     = -Pos[0];
   MATRIX_INV_TRANSL(M)[1] =  Pos[1];
   MATRIX_TRANSL(M)[1]     = -Pos[1];
   MATRIX_INV_TRANSL(M)[2] =  Pos[2];
   MATRIX_TRANSL(M)[2]     = -Pos[2];
   nA_V_Eq( MATRIX_TRANSL(M), M );
}

//////////////////////////////////////////////////////////////
//           simple camera and cube transformation          //
//////////////////////////////////////////////////////////////

static float  Roll = 0.0, Pitch = 0.0, Yaw = 0.0;
static VECTOR Position = { 0.0, 0.0, 0.0 };
static VECTOR Screen_Vo;
static float  Xc, Yc;
static float  Focal;

static MATRIX2 Mo;      // this is the final transform matrix

static float Cube_Rot = 0.0, Cube_Rot2 = 0.0;
static VECTOR Axis_Rot = { 0.23, 0.76, 0.21 };
static VECTOR Axis_Rot2 = { -0.53, 0.12, 0.67 };

//////////////////////////////////////////////////////////////

static void Setup_Transform( int Do_Move, int Do_Scale )
{
   MATRIX2 M;

      // setup some transform for the cube

   if ( Do_Move ) { Cube_Rot += 0.01; Cube_Rot2 += 0.02123; }

   Axis_Rotation( Axis_Rot, Cube_Rot, M );
   Axis_Rotation( Axis_Rot2, Cube_Rot2, Mo );
   Mult_Matrix2_BA( M, Mo );

      // add a scale transform, so we don't always have
      // an isometry

   if ( Do_Scale )
   {
      float Sx, Sy, Sz;
      static float Phi = 0.0;

      Sx = 1.0 + 0.5*cos( Phi );
      Sy = 1.0 + 0.5*sin( 1.1*Phi );
      Sz = 1.0;
      Phi += .02;
      Scale_Matrix( Mo, Sx, Sy, Sz );
      Mult_Matrix2_BA( M, Mo );
   }

     // compose with camera's one

   Camera_RYP( Yaw, Pitch, Roll, Position, Mo );
   Mult_Matrix2_BA( Mo, M );
}

static void Init_Positions( )
{
   Position[0] =   0.0;
   Position[1] =   0.0;
   Position[2] = -70.0;
   Roll = 0.0;
   Pitch = 0.0;
   Yaw = 0.0;

   Cube_Rot  = 0.0;
   Cube_Rot2 = 0.0;

   Xc = The_W/2.0f;
   Yc = The_H/2.0f;
   Focal = The_W;          // fov = 45'
}

//////////////////////////////////////////////////////////////
